/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/***********************************************************************/
/* File seq_bool.c                                                         */
/* functions for logical and arithmetic operations with boolean fctns  */
/* 1997 Robert Siegmund                                                */
/***********************************************************************/

#include "xseqbool.h"
#include <stdlib.h> 
#include <xtypedef.h>
#include <xutil.h>
/*#include <bios_malloc.h> */
/*#include <fwutil.h> */

/***********************************************************************/
/*  Function alloc_boolmap()                                           */
/*  returns the boolean function map of the given variable             */
/*  1997 Robert Siegmund                                               */
/***********************************************************************/

extern bx_errtype yaccerr;
 
boolmap * alloc_boolmap(no_of_vars, varpos)
     int no_of_vars;  /* number of variables of the function */
     int varpos;  /* the position of the variable */
                  /* in the sequencer input set   */
{
  int i,j,boolmap_size;
  boolmap * boolmap_ptr;

  int period; /* the returned map is of the form  */
              /* '0(n)1(n)0(n)1(n)...'            */
              /* where n is the period            */

  if(varpos >= no_of_vars && varpos != 254 && varpos != 255)
     {
       yaccerr = BX_E_SEQ_UNKNOWN_TOKEN;
       /*error_msg("alloc_boolmap: Variable Index not in Input Set");*/
       return 0;
     }

  if(varpos != 254 && varpos != 255)
    period = 1 << varpos;               /* 2^varpos */
  else
    period = 1;
 
  boolmap_size = 1 << no_of_vars; 

  if(boolmap_size < 8)
      boolmap_size = 8;   /* we need at least 1 character, even if */
                          /* some bits are not used                */

  boolmap_ptr=(boolmap*) BestXMemMalloc((size_t)sizeof(boolmap));
  if (boolmap_ptr==NULL)
  {
    yaccerr = BX_E_HOST_MEM_FULL;
    return NULL;
  }

  boolmap_ptr->map_ptr=BestXMemMalloc((size_t)(boolmap_size/8));
  if (boolmap_ptr->map_ptr==NULL)
  {
    yaccerr = BX_E_HOST_MEM_FULL;
    return NULL;
  }

  boolmap_ptr -> map_size = boolmap_size;
  boolmap_ptr -> fcn_vars = no_of_vars;
  
  for(i=0; i < boolmap_size/8; i++)
    {
      *((boolmap_ptr -> map_ptr) + i) = 0;

      for(j=0; j < 8; j++)
        {
          if(varpos == 255)   /* make all '1' vector */
            *((boolmap_ptr -> map_ptr) + i) |= (1 << j);
          else       
          if((varpos != 254) && (((i*8+j)/period)%2 == 1))  
            *((boolmap_ptr -> map_ptr) + i) |= (1 << j);
        }
    }
  return boolmap_ptr;
}

/***********************************************************************/
/*  Function dealloc_boolmap()                                         */
/*  deallocates the boolean map from memory                            */
/*  1997 Robert Siegmund                                               */
/***********************************************************************/

void dealloc_boolmap(boolmap_ptr)
     boolmap * boolmap_ptr;
{
  if(boolmap_ptr) /* do it only if we have a legal boolmap_ptr */
  {
    BestXMemFree((void**)&(boolmap_ptr->map_ptr));
    BestXMemFree((void**)&(boolmap_ptr));
  }
}


/***********************************************************************/
/*  Function not_boolmap()                                             */
/*  returns the logical not of the given boolean function map          */
/***********************************************************************/

boolmap * not_boolmap(boolmap * boolmap_ptr)
{
  int i;

  boolmap * result_ptr;

  if(boolmap_ptr == NULL)
  {
    yaccerr = BX_E_SEQ_SYNTAX_ERR;
    return NULL;
  }

  result_ptr = alloc_boolmap(boolmap_ptr -> fcn_vars,0);

  for(i=0; (i < (boolmap_ptr -> map_size)/8) && result_ptr; i++)
    {
      *((result_ptr -> map_ptr) + i) = \
             ~*((boolmap_ptr -> map_ptr) + i);
    }

  return result_ptr;
}

/***********************************************************************/
/*  Function and_boolmap()                                             */
/*  returns the logical and of the given boolean function map          */
/***********************************************************************/

boolmap * and_boolmap(boolmap0_ptr, boolmap1_ptr)
     boolmap * boolmap0_ptr;
     boolmap * boolmap1_ptr;
{
  int i;
  boolmap * result_ptr = NULL;

  if((boolmap0_ptr == NULL) || (boolmap0_ptr == NULL) || 
     (boolmap0_ptr -> map_size != boolmap1_ptr -> map_size))
  {
    /*error_msg("and_boolmap: Size Mismatch in Boolean Function Maps");*/
    yaccerr = BX_E_SEQ_SYNTAX_ERR;
  }
  
  else
  {
    result_ptr = alloc_boolmap(boolmap0_ptr -> fcn_vars,0);
    for(i=0; (i < (boolmap0_ptr -> map_size)/8) && result_ptr; i++)
    {
      *((result_ptr -> map_ptr) + i) = \
         *((boolmap0_ptr -> map_ptr) + i)\
& *((boolmap1_ptr -> map_ptr) + i);
    }
  }
  return result_ptr;
}

/***********************************************************************/
/*  Function or_boolmap()                                              */
/*  returns the logical and of the given boolean function map          */
/***********************************************************************/

boolmap * or_boolmap(boolmap0_ptr, boolmap1_ptr)
     boolmap * boolmap0_ptr;
     boolmap * boolmap1_ptr;
{
  int i;
  boolmap * result_ptr = NULL;

  if((boolmap0_ptr == NULL) || (boolmap0_ptr == NULL) || 
     (boolmap0_ptr -> map_size != boolmap1_ptr -> map_size))
  {
    /*error_msg("and_boolmap: Size Mismatch in Boolean Function Maps");*/
    yaccerr = BX_E_SEQ_SYNTAX_ERR;
  }
  
  else
    {
      result_ptr = alloc_boolmap(boolmap0_ptr -> fcn_vars,0);
      for(i=0; (i < (boolmap0_ptr -> map_size)/8) && result_ptr; i++)
        {
          *((result_ptr -> map_ptr) + i) = \
                      *((boolmap0_ptr -> map_ptr) + i)\
                    | *((boolmap1_ptr -> map_ptr) + i);
        }
    }
  return result_ptr;
}

/***********************************************************************/
/*  Function xor_boolmap()                                             */
/*  returns the logical and of the given boolean function map          */
/***********************************************************************/

boolmap * xor_boolmap(boolmap0_ptr, boolmap1_ptr)
     boolmap * boolmap0_ptr;
     boolmap * boolmap1_ptr;
{
  int i;
  boolmap * result_ptr = NULL;

  if((boolmap0_ptr == NULL) || (boolmap0_ptr == NULL) || 
     (boolmap0_ptr -> map_size != boolmap1_ptr -> map_size))
  {
    /*error_msg("and_boolmap: Size Mismatch in Boolean Function Maps");*/
    yaccerr = BX_E_SEQ_SYNTAX_ERR;
  }
  
  else
    {
      result_ptr = alloc_boolmap(boolmap0_ptr -> fcn_vars,0);
      for(i=0; (i < (boolmap0_ptr -> map_size)/8) && result_ptr; i++)
        {
          *((result_ptr -> map_ptr) + i) = \
                      *((boolmap0_ptr -> map_ptr) + i)\
                    ^ *((boolmap1_ptr -> map_ptr) + i);
        }
     }
   return result_ptr;
}

/***********************************************************************/
/*  Function cmp_boolmap()                                             */
/*  compares the given boolean function maps                           */
/***********************************************************************/

int cmp_boolmap(boolmap * boolmap0_ptr,boolmap * boolmap1_ptr)
{
  int i;

  /* check the incoming parameters first */
  if(boolmap0_ptr == NULL &&  boolmap1_ptr == NULL)
    return 1;

  if(boolmap0_ptr == NULL ||  boolmap1_ptr == NULL)
    return 0;

  if(boolmap0_ptr -> map_size != boolmap1_ptr -> map_size)
  {
    /*error_msg("and_boolmap: Size Mismatch in Boolean Function Maps");*/
    yaccerr = BX_E_SEQ_SYNTAX_ERR;
  }
  
  else
    {
      for(i=0; i < (boolmap0_ptr -> map_size)/8; i++)
        {
          if(*((boolmap0_ptr -> map_ptr) + i) != \
             *((boolmap1_ptr -> map_ptr) + i))
                 return(0);
        }
     }
   return(1);
}

/***********************************************************************/
/*  Function add_boolmap()                                             */
/*  returns the arithmetic sum of the given boolean function maps      */
/***********************************************************************/

boolmap * add_boolmap(boolmap * boolmap0_ptr, boolmap * boolmap1_ptr)
{
  int i;
  int sum;
  int ov = 0;

  boolmap * result_ptr = NULL;
  
  if((boolmap0_ptr == NULL) || (boolmap1_ptr == NULL) ||
     (boolmap0_ptr -> map_size != boolmap1_ptr -> map_size))
  {
    /*error_msg("and_boolmap: Size Mismatch in Boolean Function Maps");*/
    yaccerr = BX_E_SEQ_SYNTAX_ERR;
  }
  
  else
    {
      result_ptr = alloc_boolmap(boolmap0_ptr -> fcn_vars,0);

      for(i=0; (i < (boolmap0_ptr -> map_size)/8) && result_ptr; i++)
        {
          sum = (*((boolmap0_ptr -> map_ptr) + i) & 0xFF) + \
                (*((boolmap1_ptr -> map_ptr) + i) & 0xFF) + ov;
          *((result_ptr -> map_ptr) + i) = (sum & 0xFF);
      
          if(sum > 0xFF)
             ov = 1;
          else
             ov = 0;
        }
    }
  return result_ptr;
}

/***********************************************************************/
/*  Function get_boolmap_bit()                                         */
/*  returns the bit of the given position in the boolmap               */
/***********************************************************************/

int get_boolmap_bit(boolmap * boolmap_ptr,int bitpos)
{
  int N, R;
  char tmp;

  if((boolmap_ptr == NULL) || ((boolmap_ptr -> map_size) <= bitpos))
  {
    /*error_msg("get_bit_from_boolmap: bit index out of range");*/
    yaccerr = BX_E_SEQ_SYNTAX_ERR;
  }
  else
    {
      N = bitpos / 8;
      R = bitpos % 8;
      tmp = *((boolmap_ptr -> map_ptr) + N);
      return (tmp >> R) & 0x01;
    } 
  return 0;
}


/***********************************************************************/
/*  display_boolmap()                                                  */
/*  prints the given boolean function map                              */
/***********************************************************************/
#if 0
void display_boolmap(boolmap * boolmap_ptr, char format)
{
  int i,j,k,boolmap_size;

  j=7;

  if(boolmap_ptr == NULL)
  {
    printf("Empty boolmap, cannot print anything, sorry.\n");
    return;
  }

  boolmap_size = boolmap_ptr -> map_size;
  k=boolmap_size / 8 - 1;
  
  if(k < 0) k = 0;

  for(i=0; i < boolmap_size;)
    {
      if(format == 'H')
        {
          printf("%02x",(char)*((boolmap_ptr -> map_ptr) + k) & 0xFF);
          k --;
          i += 8;
        }
      else if(format == 'B')
        {
          printf("%1x",((*((boolmap_ptr -> map_ptr)+k) >> j) & 0x01));
          i++;
          if(i%8 == 0 && i != 0)
           {
             j = 7;
             k --;
           }
          else
             j --;
        }
    }
      printf("\n");
}
#endif
